;; jabber.el - a minimal jabber instant messaging client

;; Copyright (C) 2002 - tom berger - object@intelectronica.net

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

(require 'xml)

(require 'rx)

(defvar *jabber-connection* nil
  "the process that does the actual connection")

(defvar *jabber-roster* nil
  "the roster list")

(defvar *jabber-active-groupchats* nil
  "a list of the groupchats we are currently in")

(defvar *jabber-open-info-queries* nil
  "an alist of open query id and their callback functions")

(defvar *jabber-connected* nil
  "boolean - are we connected")

(defvar *jabber-current-status* "na"
  "the users current presence staus")

(defvar *jabber-current-show* ""
  "the user's current presence show")

(defvar *jabber-xmlq* ""
  "a string with all the incoming xml that is waiting to be parsed")

(defgroup jabber nil "Jabber instant messaging"
  :group 'emacs)

(defgroup jabber-faces nil "faces for displaying jabber instant messaging"
  :group 'jabber)

(defface jabber-title-small
  '((t (:foreground "black" :weight bold :width semi-expanded :height 1.0)))
  "face for displaying online users"
  :group 'jabber-faces)

(defface jabber-title-medium
  '((t (:foreground "black" :weight bold :width expanded :height 2.0)))
  "face for displaying online users"
  :group 'jabber-faces)

(defface jabber-title-large
  '((t (:foreground "black" :weight bold :width ultra-expanded :height 3.0)))
  "face for displaying online users"
  :group 'jabber-faces)

(defface jabber-roster-user-online
  '((t (:foreground "blue" :weight bold :slant normal)))
  "face for displaying online users"
  :group 'jabber-faces)

(defface jabber-roster-user-xa
  '((t (:foreground "black" :weight normal :slant italic)))
  "face for displaying extended away users"
  :group 'jabber-faces)

(defface jabber-roster-user-dnd
  '((t (:foreground "red" :weight normal :slant italic)))
  "face for displaying do not disturb users"
  :group 'jabber-faces)

(defface jabber-roster-user-away
  '((t (:foreground "dark green" :weight normal :slant italic)))
  "face for displaying away users"
  :group 'jabber-faces)

(defface jabber-roster-user-chatty
  '((t (:foreground "dark orange" :weight bold :slant normal)))
  "face for displaying chatty users"
  :group 'jabber-faces)

(defface jabber-roster-user-offline
  '((t (:foreground "grey" :weight light :slant italic)))
  "face for displaying offline users"
  :group 'jabber-faces)

(defface jabber-chat-prompt-local
  '((t (:foreground "blue" :weight bold)))
  "face for displaying the chat prompt for what you type in"
  :group 'jabber-faces)

(defface jabber-chat-prompt-foreign
  '((t (:foreground "red" :weight bold)))
  "face for displaying the chat prompt for what they send"
  :group 'jabber-faces)

(defcustom jabber-debug nil 
  "show debugging information." 
  :type 'boolean
  :group 'jabber)

(defcustom jabber-server "magaf.org" 
  "jabber server" 
  :type 'string
  :group 'jabber)

(defcustom jabber-username "emacs"
  "jabber username" 
  :type 'string
  :group 'jabber)

(defcustom jabber-password "emacs"
  "jabber password" 
  :type 'string
  :group 'jabber)

(defcustom jabber-resource "emacs"
  "jabber resource" 
  :type 'string
  :group 'jabber)

(defcustom jabber-port 5222
  "jabber port" 
  :type 'string
  :group 'jabber)

(defcustom jabber-nickname "emacs"
  "jabber nickname" 
  :type 'string
  :group 'jabber)

(defgroup jabber-alerts nil "auditory and visual alerts for jabber events"
  :group 'jabber)

(defcustom jabber-alert-on-event t
  "make a sound when a new message, invitation or subscription request arrives"
  :type 'boolean
  :group 'jabber-alerts)

(defcustom jabber-alert-message-wave nil
  "a sound file to play when a message arrived"
  :type 'file
  :group 'jabber-alerts)

(defcustom jabber-alert-presence-wave nil
  "a sound file to play when a presence arrived"
  :type 'file
  :group 'jabber-alerts)

(defcustom jabber-alert-info-wave nil
  "a sound file to play when an info query result arrived"
  :type 'file
  :group 'jabber-alerts)

(defvar menu-bar-jabber-menu (make-sparse-keymap "jabber-menu"))

(define-key global-map
  [menu-bar menu-bar-jabber-menu]
  (cons "Jabber" menu-bar-jabber-menu))

(define-key menu-bar-jabber-menu
  [jabber-menu-display-roster]
  '("JABBER" . jabber))

(define-key-after menu-bar-jabber-menu
  [jabber-menu-seperator-1]
  '("--")
  'jabber-menu-display-roster)

(define-key-after menu-bar-jabber-menu
  [jabber-menu-connect]
  '("Connect" . jabber-connect)
  'jabber-menu-seperator-1)

(define-key-after menu-bar-jabber-menu
  [jabber-menu-disconnect]
  '("Disconnect" . jabber-disconnect)
  'jabber-menu-connect)

(define-key-after menu-bar-jabber-menu
  [jabber-menu-customize]
  '("Customize" . jabber-customize)
  'jabber-menu-disconnect)

(define-key-after menu-bar-jabber-menu
  [jabber-menu-seperator-2]
  '("--")
  'jabber-menu-customize)

(define-key-after menu-bar-jabber-menu
  [jabber-menu-status]
  (cons "Set Status" (make-sparse-keymap "set-status"))
  'jabber-menu-seperator-2)

(defmacro jabber-define-status-key (title show)
  "a helper macro for defining the -set-status- menu items"
  (list 'let (list ( list 'func (list 'make-symbol (list 'concat "jabber-send-presence-" show)))
         (list 'menu-item (list 'make-symbol (list 'concat "jabber-menu-status-" show))))
     (list 'fset 'func `'(lambda () (interactive)
                           (jabber-send-presence ,show (read-string "status: "))))
     (list 'define-key 'global-map
           (list 'vector ''menu-bar ''menu-bar-jabber-menu ''jabber-menu-status 'menu-item)
           (list 'cons title 'func))))

(define-key-after menu-bar-jabber-menu
  [jabber-menu-browse]
  '("Browse" . jabber-send-browse)
  'jabber-menu-status)

(define-key-after menu-bar-jabber-menu
  [jabber-menu-seperator-3]
  '("--")
  'jabber-menu-browse)

(define-key-after menu-bar-jabber-menu
  [jabber-menu-groupchat-join]
  '("Join Groupchat" . jabber-groupchat-join)
  'jabber-menu-seperator-3)

(define-key-after menu-bar-jabber-menu
  [jabber-menu-groupchat-leave]
  '("Leave Groupchat" . jabber-groupchat-leave)
  'jabber-menu-groupchat-join)

(jabber-define-status-key "Online" "")
(jabber-define-status-key "Away" "away")
(jabber-define-status-key "Extended Away" "xa")
(jabber-define-status-key "Do not Disturb" "dnd")
(jabber-define-status-key "Unavailable" "na")

(defun jabber ()
  "display the jabber buffer"
  (interactive)
  (if *jabber-connected*
      (jabber-display-roster)
    (message "Jabber is not active. call jabber-connect to activate")))

(defun jabber-customize ()
  "customize jabber options"
  (interactive)
  (customize-group 'jabber))

(defun jabber-escape-xml (str)
  "escape strings for xml"
  str)

(defun jabber-unescape-xml (str)
  "unescape xml strings"
  str)

(defun jabber-play-sound-file (soundfile)
  (run-with-idle-timer 0.01 nil 
                       (lambda (sf)
                         (condition-case nil
                             (play-sound-file sf)
                           (error nil)))
                       soundfile))

(defun jabber-alert (event)
  (and jabber-alert-on-event
       (cond
        ((and (eq event 'message)
              jabber-alert-message-wave
              (jabber-play-sound-file jabber-alert-message-wave)))
        ((and (eq event 'presence)
              jabber-alert-presence-wave
              (jabber-play-sound-file jabber-alert-presence-wave)))
        ((and (eq event 'info)
              jabber-alert-info-wave
              (jabber-play-sound-file jabber-alert-info-wave)))
        (t
         (beep t)))))

(defun sexp2xml (sexp)
  "converts an SEXP in the format (tagname ((attribute-name . attribute-value)...) children...) and converts it to well-formatted xml."
  (cond
   ((stringp sexp)
    sexp)
   ((listp (car sexp))
    (let ((xml ""))
      (dolist (tag sexp)
	(setq xml (concat xml (sexp2xml tag))))
      xml))
   (t
    (let ((xml ""))
      (setq xml (concat "<" 
			(symbol-name (car sexp))))
      (dolist (attr (cadr sexp))
	(if (consp attr)
	    (setq xml (concat xml
			      (format " %s='%s'"
				      (symbol-name (car attr))
				      (cdr attr))))))
      (if (cddr sexp)
	  (progn
	    (setq xml (concat xml ">"))
	    (dolist (child (cddr sexp))
	      (setq xml (concat xml
				(sexp2xml child))))
	    (setq xml (concat xml
			      "</"
			      (symbol-name (car sexp))
			      ">")))
	(setq xml (concat xml
			  "/>")))
      xml))))

(defun jabber-send-sexp (sexp)
  "send the xml corresponding to SEXP to the jabber server"
  (process-send-string *jabber-connection* (sexp2xml sexp)))

(defun jabber-read-jid-completing (prompt)
  "read a jid out of the curent roster from the minibuffer."
  (completing-read prompt
		   (mapcar (lambda (item) (cons (symbol-name item) nil))
			   *jabber-roster*)))

(defun jabber-browse-mode ()
"\\{jabber-browse-mode-map}"
  (setq major-mode 'jabber-browse-mode
        mode-name "jabber-browse")
  (use-local-map jabber-browse-mode-map)
  (setq buffer-read-only t))

(put 'jabber-browse-mode 'mode-class 'special)

(defvar jabber-browse-mode-map (make-keymap))

(suppress-keymap jabber-browse-mode-map)

(defun jabber-groupchat-mode ()
  "\\{jabber-groupchat-mode-map}"
  (make-local-variable 'jabber-group)
  (setq major-mode 'jabber-groupchat-mode
        mode-name "jabber-groupchat")
  (use-local-map jabber-groupchat-mode-map)
  (setq buffer-read-only t))

(put 'jabber-groupchat-mode 'mode-class 'special)

(defvar jabber-groupchat-mode-map (make-keymap))

(suppress-keymap jabber-groupchat-mode-map)

(dolist (key (append "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890`~!@#$%^&*()_+-=[]{}|';:/?.,>< " nil))
  (let ((send-fun (make-symbol (concat "jabber-groupchat-buffer-send-" (char-to-string key)))))
    (fset send-fun `(lambda (body) (interactive (list (read-string "" ,(char-to-string key))))
		      (jabber-send-groupchat jabber-group body)
		      (setq buffer-read-only nil)
		      (goto-char (point-max))
		      (jabber-groupchat-mode)))
    (define-key jabber-groupchat-mode-map (char-to-string key) send-fun)))

(defun jabber-groupchat-display (group &optional nick body)
  "display the groupchat window and an incoming message, if there is one"
  (with-current-buffer (get-buffer-create (concat "*-jabber-groupchat-:-" group "-*"))
    (goto-char (point-max))
    (setq buffer-read-only nil)
    (if body (insert (propertize (concat "[" (substring (current-time-string) 11 16) "] " nick)
                                 'face 'jabber-chat-prompt-foreign)
                     "> " body "\n"))
    (jabber-groupchat-mode)
    (setq jabber-group group)
    (switch-to-buffer (current-buffer)))
  (jabber-alert 'message))

(defun jabber-chat-mode ()
  "\\{jabber-chat-mode-map}"
  (make-local-variable 'jabber-chatting-with)
  (setq major-mode 'jabber-chat-mode
        mode-name "jabber-chat")
  (use-local-map jabber-chat-mode-map))

(put 'jabber-chat-mode 'mode-class 'special)

(defvar jabber-chat-mode-map (make-keymap))

(suppress-keymap jabber-chat-mode-map)


(dolist (key (append "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890`~!@#$%^&*()_+-=[]{}|';:/?.,>< " nil))
  (let ((send-fun (make-symbol (concat "jabber-chat-buffer-send-" (char-to-string key)))))
    (fset send-fun `(lambda (body) (interactive (list (read-string "" ,(char-to-string key))))
		      (jabber-send-chat jabber-chatting-with body)
		      (setq buffer-read-only nil)
		      (goto-char (point-max))
		      (insert (propertize (concat "[" (substring (current-time-string) 11 16) "] " jabber-username)
                                          'face 'jabber-chat-prompt-local) "> " body "\n")
		      (jabber-chat-mode)))
    (define-key jabber-chat-mode-map (char-to-string key) send-fun)))

(defun jabber-chat-display (&optional from body)
  "display the chat window and a new message, if there is one"
  (with-current-buffer (get-buffer-create (concat "*-jabber-chat-:-" (jabber-jid-username from) "-*"))
    (goto-char (point-max))
    (setq buffer-read-only nil)
    (if body (insert (propertize (concat "[" (substring (current-time-string) 11 16) "] " (jabber-jid-username from))
                                 'face 'jabber-chat-prompt-foreign)
                     "> " body "\n"))
    (jabber-chat-mode)
    (setq jabber-chatting-with (jabber-jid-user from))
    (switch-to-buffer (current-buffer)))
  (jabber-alert 'message))

(defun jabber-chat-with (jid)
  "open an empty chat window for chatting with JID"
  (interactive (list (jabber-read-jid-completing "chat with:")))
  (jabber-chat-display (concat jid "/chat") nil))

(defun jabber-jid-username (string)
  "return the username portion of a JID"
  (string-match (rx (and (group (* anything)) "@" anything "/" (* anything)))
                string)
  (match-string 1 string))

(defun jabber-jid-user (string)
  "return the user (username@server) portion of a JIF"
  (string-match (rx (and (* anything) "@" (* (not (in "/")))))
                string)
  (match-string 0 string))

(defun jabber-jid-resource (string)
  "return the resource portion of a JID"
  (string-match (rx (and (* anything) "@" (* anything) "/" (group (* anything))))
                string)
  (match-string 1 string))

(defun jabber-process-message (from subject body thread type)
  "process incoming messages"
  (cond
   ((string= type "groupchat")
    (jabber-groupchat-display (jabber-jid-user from) 
                              (jabber-jid-resource from)
                              (jabber-unescape-xml body))
    )
   (t
    (jabber-chat-display from 
                         (jabber-unescape-xml body)))))


(defun jabber-process-subscription-request (from presence-status)
  "process an incoming subscription request"
  (jabber-alert 'presence)
  (jabber-send-sexp 
   (list 'presence (list (cons 'to from)
			 (cons 'type (if (y-or-n-p (format "the user  - %s -  has requested to subscribe to your presence (%s). allow?" 
							   from
							   (jabber-unescape-xml presence-status)))
					 "subscribed"
				       "unsubscscribed"))))))

(defun jabber-process-presence (from to presence-show presence-status type)
  "process incoming presence tags"
  (cond
   ((string= type "subscribe")
    (run-with-idle-timer 0.01 nil #'jabber-process-subscription-request from presence-status))
   
   (t
    (dolist (buddy *jabber-roster*)
      (if (string= (symbol-name buddy) (jabber-jid-user from))
	  (progn
	    (put buddy 'show 
                 (jabber-unescape-xml presence-show))
	    (put buddy 'status 
                 (jabber-unescape-xml presence-status)))))))
  (jabber-display-roster))

(defun jabber-display-roster ()
  "switch to the main jabber buffer and refresh the roster display to reflect the current information"
  (with-current-buffer (process-buffer *jabber-connection*)
    (erase-buffer)
    (insert (propertize jabber-server 'face 'jabber-title-large) "\n__________________________________\n\n")
    (let ((map (make-sparse-keymap)))
      (define-key map [mouse-2] #'jabber-send-presence)
      (insert (propertize (format " - %s (%s) -"
                                  (cond
                                   ((string= *jabber-current-show* "na")
                                    "Disconnected")
                                   ((string= *jabber-current-show* "xa")
                                    "Extended Away")
                                   ((string= *jabber-current-show* "away")
                                    "Away")
                                   ((string= *jabber-current-show* "dnd")
                                    "Do not Disturb")
                                   ((string= *jabber-current-show* "chat")
                                    "Chatty")
                                   (t
                                    "Online"))
                                  *jabber-current-status*)
                          'face (cond
                                 ((string= *jabber-current-show* "away")
                                  'jabber-roster-user-away)
                                 ((string= *jabber-current-show* "xa")
                                  'jabber-roster-user-xa)
                                 ((string= *jabber-current-show* "dnd")
                                  'jabber-roster-user-dnd)
                                 ((string= *jabber-current-show* "chat")
                                  'jabber-roster-user-chatty)
                                 (t
                                  'jabber-roster-user-online))
                          'mouse-face (cons 'background-color "light grey")
                          'keymap map)
              "\n__________________________________\n\n"))
    (dolist (buddy *jabber-roster*)
      (let ((buddy-str (concat (if (and (member 'show (symbol-plist buddy))
					(not (string= (get buddy 'status) "Disconnected")))
				   " * "
				 "   ")
			       (if (> (length (get buddy 'name)) 0)
				   (get buddy 'name)
				 (symbol-name buddy))
			       (if (get buddy 'show)
				   (format " - %s" (get buddy 'show))
				 "")
			       (if (get buddy 'status)
				   (format " (%s)" (get buddy 'status))
				 ""))))
	(put-text-property 0
			   (length buddy-str)
			   'face
			   (if (and (member 'show (symbol-plist buddy))
                                    (not (string= (get buddy 'status) "Disconnected")))
                               (cond
                                ((string= (get buddy 'show) "away")
                                 'jabber-roster-user-away)
                                ((string= (get buddy 'show) "xa")
                                 'jabber-roster-user-xa)
                                ((string= (get buddy 'show) "dnd")
                                 'jabber-roster-user-dnd)
                                ((string= (get buddy 'show) "chat")
                                 'jabber-roster-user-chatty)
                                (t
                                 'jabber-roster-user-online))
                             'jabber-roster-user-offline)
			   buddy-str)
	(put-text-property 0
			   (length buddy-str)
			   'mouse-face
			   (cons 'background-color "light grey")
			   buddy-str)
	(let ((map (make-sparse-keymap))
	      (chat-with-func (make-symbol (concat "jabber-chat-with" (symbol-name buddy)))))
	  (fset chat-with-func `(lambda () (interactive) (jabber-chat-with ,(symbol-name buddy))))
	  (define-key map [mouse-2] chat-with-func)
	  (put-text-property 0
			     (length buddy-str)
			     'keymap 
			     map
			     buddy-str))
	(insert buddy-str "\n\n")))
    (insert "__________________________________")
    (switch-to-buffer (current-buffer)))
  (jabber-alert 'info))

(defun jabber-process-roster (xml-data)
  "process an incoming roster infoquery result"
  (dolist (item (xml-get-children (car (xml-get-children xml-data 'query)) 'item))
    (setq *jabber-roster* (cons (make-symbol (xml-get-attribute item 'jid)) *jabber-roster*))
    (put (car *jabber-roster*) 'name (xml-get-attribute item 'name))
    (put (car *jabber-roster*) 'subscription (xml-get-attribute item 'subscription)))
  (jabber-display-roster))

(defun jabber-process-browse (xml-data)
  "process and incoming browse infoquery result"
  (with-current-buffer (get-buffer-create (concat "*-jabber-browse-:-" from "-*"))
    (setq buffer-read-only nil)
    (insert (propertize (xml-get-attribute xml-data 'from) 
                        'face 'jabber-title-large) "\n\n")
    (dolist (item (xml-node-children xml-data))
      (cond
       ((eq (xml-node-name item) 'user)
        (insert (propertize "$ USER"
                            'face 'jabber-title-medium)
                "\n\n"))
       ((eq (xml-node-name item) 'service)
        (insert (propertize "* SERVICE"
                            'face 'jabber-title-medium)
                "\n\n"))
       ((eq (xml-node-name item) 'conference)
        (insert (propertize "@ CONFERENCE"
                            'face 'jabber-title-medium)
                "\n\n"))
       )
      ;; debug information
      (insert (format "%S" xml-data)))
    (jabber-browse-mode)
    (switch-to-buffer (current-buffer))
    (jabber-alert 'info)))

(defun jabber-process-logon (xml-data)
  (cond 
   ((string= (xml-get-attribute xml-data 'type) "result")
    (jabber-send-iq jabber-server
                    "get" 
                    '(query ((xmlns . "jabber:iq:roster")))
                    #'jabber-process-roster)
    (jabber-send-sexp '((presence ((type . "available")))))
    (jabber-send-presence "" "Online"))
   ((string= (xml-get-attribute xml-data 'type) "error")
    (message "error connection to jabber server"))))

(defun jabber-process-iq (xml-data)
  (let* ((id (xml-get-attribute xml-data 'id))
         (type (xml-get-attribute xml-data 'type))
         (from (xml-get-attribute xml-data 'from))
         (callback (cdr (assoc id *jabber-open-info-queries*))))
    (if callback
        (funcall callback xml-data))))

(defun jabber-process-input (xml-data)
  "process and incoming parsed tag"
  (let ((tag (xml-node-name xml-data)))
     (cond
      ((eq tag 'iq)
       (jabber-process-iq xml-data))
      
      ((eq tag 'message)
       (let ((from (xml-get-attribute xml-data 'from))
             (type (xml-get-attribute xml-data 'type))
             (subject (if (xml-get-children xml-data 'subject)
                          (car (xml-node-children (car (xml-get-children xml-data 'subject))))))
             (body (if (xml-get-children xml-data 'body)
                       (car (xml-node-children (car (xml-get-children xml-data 'body))))))
             (thread (if (xml-get-children xml-data 'thread)
			 (car (xml-node-children (car (xml-get-children xml-data 'thread)))))))
         (jabber-process-message from subject body thread type)))

      ((eq tag 'presence)
       (let ((from (xml-get-attribute xml-data 'from))
             (to (xml-get-attribute xml-data 'to))
             (type (xml-get-attribute xml-data 'type))
             (show (if (listp (car (xml-node-children xml-data)))
                       (car (xml-node-children (car (xml-get-children xml-data 'show))))))
             (status (if (listp (car (xml-node-children xml-data)))
                         (car (xml-node-children (car (xml-get-children xml-data 'status)))))))
         (jabber-process-presence from to show status type))))))
   

(defun jabber-filter (process string)
  "the filter function for the jabber process"
  (cond
   ((string-match "</stream:stream>" string)
    (jabber-disconnect))
   ((string-match "<stream:stream" string)
    (setq jabber-session-id
          (progn (string-match "id='\\([A-Za-z0-9]+\\)'" string)
               (match-string 1 string))))
   (t
    (if (active-minibuffer-window)
        (run-with-idle-timer 0.01 nil #'jabber-filter process string)
      (setq *jabber-xmlq* (concat *jabber-xmlq* string))
      (if (string-match " *\\w+=''" *jabber-xmlq*)
          (setq *jabber-xmlq* (replace-match "" nil t *jabber-xmlq*)))
      (catch 'jabber-no-tag
        (while (string-match "<\\([a-zA-Z0-9\:]+\\)\\s-" *jabber-xmlq*)
          (if (or (string-match (concat "<" (match-string 1 *jabber-xmlq*) "[^<>]*?/>") *jabber-xmlq*)
                  (string-match (concat "<" (match-string 1 *jabber-xmlq*) ".*?>[^\0]+?</" (match-string 1 *jabber-xmlq*) ">") *jabber-xmlq*))
              (with-temp-buffer
                (insert (match-string 0 *jabber-xmlq*))
                (goto-char (point-min))
                (setq *jabber-xmlq* (substring *jabber-xmlq* (match-end 0)))
                (let ((xml-data (xml-parse-region (point-min)
                                                  (point-max))))
                  (if xml-data
                      (progn
                        (if jabber-debug
                            (message "%S\n" xml-data))
                        (jabber-process-input (car xml-data))))))
            (throw 'jabber-no-tag t))))))))

(defun jabber-connect ()
  "connect to the jabber server and start a jabber xml stream"
  (interactive)
  (jabber-disconnect)
  (setq *jabber-connection* (open-network-stream "jabber"
					       "*-jabber-*"
					       jabber-server
					       jabber-port))
  (set-process-filter *jabber-connection* #'jabber-filter)
  (process-send-string *jabber-connection*
		       (concat "<?xml version='1.0'?><stream:stream to='" 
                               jabber-server 
                               "' xmlns='jabber:client' xmlns:stream='http://etherx.jabber.org/streams'>"))
  (jabber-send-iq jabber-server 
                  "set"
                  `(query ((xmlns . "jabber:iq:auth"))
                          (username ()
                                    ,jabber-username)
                          (password ()
                                    ,jabber-password)
                          (resource ()
                                    ,jabber-resource))
                  #'jabber-process-logon)
  (setq *jabber-connected* t))

(defun jabber-send-groupchat (group body)
  "send a message to a groupchat"
  (jabber-send-message group body nil "groupchat"))

(defun jabber-send-chat (to body)
  "send a chat message to someone"
  (jabber-send-message to body nil "chat"))

(defun jabber-send-subscription-request (to &optional request)
  "send a subscription request to jid, showing him your request text, if specified"
  (interactive (list (jabber-read-jid-completing "to: ")
		     (read-string "request: ")))
  (jabber-send-sexp `(presence ((to . ,to)
                                (type . "subscribe"))
                               ,(if (and request (> (length request) 0))
                                   request))))

(defun jabber-send-message (to body subject type)
  "send a message tag to the server"
  (interactive (list (jabber-read-jid-completing "to: ")
		     (read-string "subject: ")
		     (read-string "body: ")
		     (read-string "type: ")))
  (jabber-send-sexp `(message ((to . ,to)
                               ,(if (> (length type) 0)
                                    `(type . ,type)))
                              ,(if (> (length subject) 0)
                                   `(subject () ,(jabber-escape-xml subject)))
                              ,(if (> (length body) 0)
                                   `(body () ,(jabber-escape-xml body))))))

(defun jabber-groupchat-leave (group)
  "leave a groupchat"
  (interactive (list (completing-read "group: "
                                      (mapcar (lambda (item) (cons item nil))
                                              *jabber-active-groupchats*))))
  (let ((lst (member group *jabber-active-groupchats*)))
    (setcar lst (cadr lst))
    (setcdr lst (cddr lst)))
  (jabber-send-sexp `(presence ((to . ,group)
                                (type . "unavailable")))))

(defun jabber-groupchat-join (group)
  "join a groupchat"
  (interactive (list (read-string "group: ")))
  (jabber-send-sexp `(presence ((to . ,group)
                                (type . "available"))))
  (jabber-send-sexp `(iq ((to . ,group)
                          (type . "set"))
                         (query ((xmlns . "jabber:iq:conference"))
                                (nick ()
                                      ,jabber-nickname))))
  (if (not (member group *jabber-active-groupchats*))
      (setq *jabber-active-groupchats* (cons group *jabber-active-groupchats*)))
  (jabber-groupchat-display group))

(defun jabber-send-presence (show status)
  "send a presence tag to the server"
  (interactive (list (completing-read "show:"
				      '(("" . nil) ("away" . nil) ("xa" . nil) ("dnd" . nil) ("chat" . nil)))
		     (read-string "status: ")))
  (setq *jabber-current-status* status)
  (setq *jabber-current-show* show)
  (jabber-send-sexp `(presence ()
                               ,(if (> (length status) 0)
                                    `(status () ,(jabber-escape-xml status)))
			       ,(if (> (length show) 0)
                                    `(show () ,(jabber-escape-xml show)))))
  (jabber-display-roster))

(defun jabber-send-iq (to type query callback)
  (let ((id (format "emacs-iq-%f" (float-time))))
    (setq *jabber-open-info-queries* (cons (cons id callback)
                                           *jabber-open-info-queries*))
    (jabber-send-sexp `(iq ((to . ,to)
                            (type . ,type)
                            (id . ,id))
                           ,query))))

(defun jabber-send-browse (to)
  "send a browse infoquery request to someone"
  (interactive (list (read-string "browse: ")))
  (jabber-send-iq to 
                  "get"
                  '(query ((xmlns . "jabber:iq:browse")))
                  #'jabber-process-browse))

(defun jabber-disconnect ()
  "disconnect from the jabber server and re-initialise the jabber package variables"
  (interactive)
  (when *jabber-connected* ;;(eq (process-status *jabber-connection*) 'open)
    (process-send-string *jabber-connection* "</stream:stream>")
    (delete-process *jabber-connection*)
    (kill-buffer (process-buffer *jabber-connection*)))
  (setq *jabber-xmlq* "")
  (setq *jabber-roster* nil)
  (setq *jabber-connected* nil)
  (setq *jabber-active-groupchats* nil))

(provide 'jabber)
